<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\Affectation;
use App\Entity\Dettes;
use App\Entity\Expertise;
use App\Entity\Facture;
use App\Entity\Facturedateretrait;
use App\Entity\Licence;
use App\Entity\Maison;
use App\Entity\PayementDette;
use App\Entity\PointOfSaleDetails;
use App\Entity\PointOfSaleHistory;
use App\Entity\Service;
use App\Entity\UserGuichet;
use App\Entity\Vehicule;
use App\Http\ApiResponse;
use App\Tools\Exporter;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;


class LicenceController extends AbstractController
{
    /**
     * @Route("/licence/new", name="new_licence")
     */
    public function register(Request $request): Response
    {
        return $this->edit('licence/new.html.twig', $request);
    }

    /**
     * @Route("/licence/{id}", name="detail_licence")
     */
    public function detail(Request $request, $id): Response
    {
        return $this->edit('licence/detail.html.twig',$request, $id);
    }

    /**
     * @Route("/licences", name="list_licences")
     */
    public function list(Request $request): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $queryString = "SELECT p.id, p.createdAt, p.prochainevisite, p.numeroPv, v.genre, v.numeroImmatriculation, c.nomClient, f.id as invoiceId FROM App\Entity\Licence p INNER JOIN p.vehicule v INNER JOIN v.client c LEFT JOIN p.factures f";
        $searchQuery = $request->get('search');
        $limit = $request->get('limit');

        if ($searchQuery) {
            $queryString .= " WHERE c.nomClient LIKE :query";
        }
        
        $query = $entityManager->createQuery($queryString . " ORDER BY p.id DESC");

        if ($searchQuery) {
            $query = $query->setParameter('query', '%'.$searchQuery.'%');
        }

        $licenses = $query->setMaxResults($limit ? intval($limit) : 200)->getResult();

        if ($request->isXMLHttpRequest()) {
            return new ApiResponse('', $licenses, []);
        }

        return $this->render('licence/list.html.twig', [
            'title' => 'Gestion des P.V',
            'breadcrumb' => [
                ['name' => 'P.V', 'path' => '/licences'],
            ],
            'licenses' => $licenses,
            'typeuser' => $this->getUser()->getType()
        ]);
    }

    /**
     * @Route("/licenses/guichetier", name="list_guichetier_licenses")
     */
    public function licenses(Request $request): JsonResponse
    {
        $licenseDate = $request->get('date');
        $searchQuery = $request->get('search');
        $em = $this->getDoctrine()->getManager();
        $licenseRepository = $em->getRepository(Licence::class);
        $expertiseRepository = $em->getRepository(Expertise::class);

        $queryBuilder = $licenseRepository->createQueryBuilder('l');
        $queryBuilder = $queryBuilder
            ->select('l.id', 'l.createdAt', 'l.numeroPv', 'v.genre', 'v.numeroImmatriculation', 'c.nomClient')
            // ->where($queryBuilder->expr()->neq('f.typeFacture', ':typeFacture'))
        ;

        if ($licenseDate) {
            $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->eq("DATE_FORMAT(l.createdAt, '%Y-%m-%d')", ':licenseDate'));
        }

        if ($searchQuery) {
            $licenseNumber = \intval($searchQuery);

            // if (\is_int($licenseNumber) && $licenseNumber > 0) {
            //     $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->orX(
            //         $queryBuilder->expr()->eq('l.id', ':searchQuery'),
            //         $queryBuilder->expr()->eq('l.id', ':licenseNumber')
            //     ));
            // } else {
                $queryBuilder = $queryBuilder->andWhere($queryBuilder->expr()->orX(
                    $queryBuilder->expr()->like('c.nomClient', ':searchQuery'), $queryBuilder->expr()->orX(
                        $queryBuilder->expr()->orX(
                            $queryBuilder->expr()->eq('l.numeroPv', ':searchQuery'),
                            $queryBuilder->expr()->eq('l.numeroPv', ':licenseNumber')
                        ),
                        $queryBuilder->expr()->like("DATE_FORMAT(l.createdAt, '%Y-%m-%d')", ':searchQuery')
                    )
                ));
            // }
        }

        if ($this->getUser()->getType() !== 'super_guichetier') {
            $queryBuilder = $queryBuilder
                // ->andWhere($queryBuilder->expr()->orX($queryBuilder->expr()->in('f.typeFacture', ['proforma', 'proforma_ctrl']), 'u = :user'))
                ->andWhere('u = :user')
                ->innerJoin('l.vehicule', 'v')
                ->innerJoin('v.client', 'c')
                ->innerJoin('l.doneby', 'u')
                ->setParameter('user', $this->getUser())
            ;
        } else {
            $affectationRepository = $em->getRepository(Affectation::class);
            $pos = $affectationRepository->findOneBy(['user' => $this->getUser()])->getPos();
            $usersIds = $affectationRepository
                ->createQueryBuilder('a')
                ->select('u.id')
                ->where('p = :pos')
                ->innerJoin('a.pos', 'p')
                ->innerJoin('a.user', 'u')
                ->setParameter('pos', $pos)
                ->getQuery()
                ->getResult()
            ;
            $queryBuilder = $queryBuilder
                ->andWhere($queryBuilder->expr()->in('u.id', ':usersIds'))
                ->leftJoin('l.vehicule', 'v')
                ->innerJoin('v.client', 'c')
                ->innerJoin('l.doneby', 'u')
                ->setParameter('usersIds', $usersIds)
            ;
        }

        if ($licenseDate) {
            $queryBuilder = $queryBuilder->setParameter('licenseDate', $licenseDate);
        }

        if ($searchQuery) {
            if (isset($licenseNumber)) {
                $queryBuilder = $queryBuilder->setParameter('licenseNumber', $licenseNumber);
                $queryBuilder = $queryBuilder->setParameter('searchQuery', $searchQuery);
            } else {
                $queryBuilder = $queryBuilder->setParameter('searchQuery', '%'.$searchQuery.'%');
            }
        }

        $licenses = $queryBuilder
            // ->setParameter('typeFacture', 'cancelled')
            ->orderBy('l.createdAt', 'DESC')
            ->getQuery()
            ->getResult()
        ;

        return new JsonResponse($licenses);
    }

    /**
     * @Route("/license_receipts/{licenseId}/print/{template_name}", name="print_license_receipt")
     */
    public function printLicenseReceipt(Request $request, EntityManagerInterface $entityManager, Exporter $exporter, $licenseId, $template_name)
    {
        $licenseRepository = $entityManager->getRepository(Licence::class);
        $license = $licenseRepository->find($licenseId);
        $fileName = \sprintf('Reçu %s %s', $license->getId(), $license->getVehicule()->getClient()->getNomClient());

        $company = $entityManager
            ->getRepository(Maison::class)
            ->createQueryBuilder('c')
            ->select('c.logo', 'c.nomSociete')
            ->getQuery()
            ->getSingleResult()
        ;

        $invoices = $entityManager->getRepository(Facture::class)
            ->createQueryBuilder('i')
            ->where('l.id = :licenseId')
            ->innerJoin('i.licence', 'l')
            ->orderBy('i.dateFact', 'DESC')
            ->setParameter('licenseId', $license->getId())
            ->getQuery()
            ->getResult()
        ;

        $vehicle = $license->getVehicule();
        // dd($invoice);

        $header = [
            'company' => $company,
            // 'vehicle' => [
            //     'genre' => $vehicle->getGenre(),
            //     'marque' => $vehicle->getMarque(),
            //     'typeOuModele' => $vehicle->getTypeOuModele(),
            //     'numeroImmatriculation' => $vehicle->getNumeroImmatriculation(),
            //     'kilometrage' => $vehicle->getKilometrage(),
            // ],
            'client' => $vehicle->getClient()->getNomClient(),
        ];

        // $options['pageFooter'] = '{DATE F j, Y}|'.$fileName.'|Page {PAGENO} of {nb}';
        $options = [
            'header' => $header,
            // 'roll58' => [58, 1 * 6 + 18],
            // 'roll80' => [80, 1 * 6 + 18],
            // 'totalCost' => \floatval($request->query->get('totalCost')),
            // 'amountPaid' => \floatval($request->query->get('amountPaid')),
        ];
        // $staticContentHeight = 30;
        // $lineHeight = 3;

        if (\count($invoices) > 0) {
            $options['totalCost'] = $invoices[0]->getTotal();
            $options['amountPaid'] = $invoices[0]->getAmountPaid();
        }

        switch ($template_name) {
            case 'roll58':
                $options['config']['format'] = [58, 92];
                $options['config']['default_font_size'] = 8;
                $options['config']['default_font'] = 'monospace';
                break;
            case 'roll80':
                $options['config']['format'] = [80, 92];
                $options['config']['default_font_size'] = 9;
                $options['config']['default_font'] = 'monospace';
                break;
            default:
                $options['config']['format'] = $template_name;
                break;
        }
        // dd($options);
        $exporter->exportPdf(
            'export/license-receipt/'.$template_name.'.html.twig',
            $license,
            [],
            $fileName,
            $options
        );
    }

    private function edit(string $template, Request $request, $id = null): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $entityRepository = $this->getDoctrine()->getRepository(Licence::class);
        $vh = $this->getDoctrine()->getRepository(Vehicule::class);

        $licence = null;

        if (isset($id)) {
            $licence = $entityRepository->find(intval($id));
        }

        if ($request->isMethod('POST')) {
            if ($request->isXMLHttpRequest()) {
                $data = json_decode(file_get_contents('php://input'), true);
            }

            if (!isset($id)) {
                $licence = new Licence();
            }

            $lastInsertedLicense = $entityRepository
                ->createQueryBuilder('l')
                ->select('l.id')
                ->orderBy('l.id', 'DESC')
                ->setMaxResults(1)
                ->getQuery()
                ->getResult()
            ;
            // $licence->setNumeroPv($request->isXMLHttpRequest() ? $data['numeroPv'] : $request->request->get('numeroPv'));
            $licence->setNumeroPv(count($lastInsertedLicense) > 0 ? (intval($lastInsertedLicense[0]['id']) + 1) : 1);
            $licence->setVehicule($vh->find(intval($request->isXMLHttpRequest() ? $data['vehicule'] : $request->request->get('vehicule'))));
            $licence->setResultat($request->request->get('resultat'));
            $licence->setVehiculeaccepte($request->request->get('accepte'));
            $licence->setCtao($request->request->get('ctao'));
            $licence->setDoneBy($this->getUser());

            if ($request->isXMLHttpRequest()) {
                $service = null;
                $serviceRepository = $entityManager->getRepository(Service::class);
                $pos = $entityManager->getRepository(UserGuichet::class)->findOneBy(['user' => $this->getUser()])->getGuichet()->getPointOfSale();

                $serviceSent = $serviceRepository->find($data['article']);

                if ($serviceSent) {
                    $amountPaid = $data['amountPaid'];
                    $pdr = $entityManager->getRepository(PointOfSaleDetails::class);
                    // creer une nouvelle facture
                    $facture = new Facture();
                    $facture->setTotal($serviceSent->getUnitPrice());
                    $facture->setAmountPaid($amountPaid);
                    $facture->setTypeFacture($amountPaid < $data['totalCost'] ? 'credit' : 'cash');
                    $facture->setEditedBy($this->getUser());
                    $facture->setLicence($licence);

                    $client = $licence->getVehicule()->getClient();
                    $facture->setClient($client ?? null);

                    $posDetails = $pdr
                        ->createQueryBuilder('pd')
                        ->select('pd')
                        ->where('s.id = :serviceId')
                        ->andWhere('pd.quantity > 0')
                        ->innerJoin('pd.service', 's')
                        ->setParameter('serviceId', intval($serviceSent->getId()))
                        ->getQuery()
                        ->getResult()
                    ;

                    if ($posDetails !== null && count($posDetails) > 0) {
                        $posDetail = $posDetails[0];
                        $service = $posDetail->getService();
                        $destockage = new PointOfSaleHistory();
                        $destockage->setMotifDestockage('vente_cash');
                        $destockage->setQuantity(-1);
                        $destockage->setPrice($service->getUnitPrice());
                        $posDetail->setQuantity($posDetail->getQuantity() - 1);
                        $posDetail->setValue($posDetail->getValue() - $service->getUnitPrice());
                        $service->addPointOfSaleHistory($destockage);
                        $pos->addPointOfSaleHistory($destockage);
                        $facture->addPointOfSaleHistory($destockage);
                        $entityManager->persist($destockage);
                    }

                    if ($facture->getTypeFacture() === 'credit' && !\is_null($facture->getClient())) {
                        $dette = new Dettes();
                        $debtAmount = $facture->getTotal();
                        if ($amountPaid > 0) {
                            $paiementDette = new PayementDette();
                            $paiementDette->setMontantPaye($amountPaid);
                            $dette->addPayementdette($paiementDette);
                            $this->getUser()->addPayementdette($paiementDette);
                            $entityManager->persist($paiementDette);
                        }
                        // $date = trim($data['date']); // $data['date'] est sous la forme '2021-10-15T19:30'
                        $entry = new Facturedateretrait();
                        // $entry->setDateRetrait(new \DateTime($date));
                        $entry->setDateRetrait(new \DateTime('+7 day'));
                        $facture->addFacturedateretrait($entry);
                        $dette->setFacture($facture);
                        $dette->setStatut($amountPaid > 0 ? 1 : 0);
                        $dette->setAmount($debtAmount);
                        $dette->setSold($debtAmount - $amountPaid);
                        $dette->setEcheance($entry->getDateRetrait());
                        $this->getUser()->addDette($dette);
                        $dette->setClient($client);
                        $entityManager->persist($entry);
                        $entityManager->persist($dette);
                    }

                    $entityManager->persist($facture);
                }
            }

            if (!isset($id)) {
                $entityManager->persist($licence);
            }

            $entityManager->flush();
            // do anything else you need here, like send an email
            $parameters = [];
            if (isset($id)) {
                $placeholder = 'modifié';
                $redirect = 'detail_licence';
                $parameters['id'] = $id;
            } else {
                $placeholder = 'créé';
                $redirect = 'new_licence';
            }

            $message = sprintf("P.V %s avec succès", $placeholder);

            if ($request->isXMLHttpRequest()) {
                $vehicule = $licence->getVehicule();
                $user = $this->getUser();
                return new JsonResponse([
                    'message' => $message,
                    'data' => [
                        'id' => $licence->getId(),
                        'createdAt' => $licence->getCreatedAt(),
                        'prochainevisite' => $licence->getProchaineVisite(),
                        'numeroPv' => $licence->getNumeroPv(),
                        'numeroImmatriculation' => $vehicule->getNumeroImmatriculation(),
                        'genre' => $vehicule->getGenre(),
                        'marque' => $vehicule->getMarque(),
                        'typeOuModele' => $vehicule->getTypeOuModele(),
                        'nomClient' => $vehicule->getClient()->getNomClient(),
                        'installment' => $data['amountPaid']
                    ],
                    'sommesent' => isset($service) && $service !== null ? $service->getUnitPrice() : 0,
                    'doneby' => $user->getFullName() ?? $user->getUserIdentifier(),
                    'createdat'=>date('d-m-Y'),
                    'errors' => []
                ]);
            }

            $this->addFlash('success', $message);

            return $this->redirectToRoute($redirect, $parameters);
        }

        return $this->render($template, [
            'title' => 'Gestion des P.V',
            'breadcrumb' => [
                ['name' => 'P.V', 'path' => '/licences'],
                ['name' => isset($id) ? $licence->getNumeroPv() : 'Nouveau P.V']
            ],
            'latestLicences' => $entityRepository->findBy([], ['numeroPv' => 'DESC']),
            'currentL' => $licence,
            'lastestVehicules'=>$vh->findBy([], ['numeroImmatriculation' => 'DESC']),
        ]);
    }
}
