<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\Currency;
use App\Http\ApiResponse;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Intl\Currencies;
use Symfony\Component\Routing\Annotation\Route;

class CurrencyController extends AbstractController
{
    /**
     * @Route("/currencies", name="currencies_list")
     */
    public function list(): Response
    {
        $entityRepository = $this->getDoctrine()->getRepository(Currency::class);

        return $this->render('currencies/list.html.twig', [
            'title' => 'Monnaies',
            'breadcrumb' => [
                ['name' => 'Monnaies']
            ],
            'currencies' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
        ]);
    }

    /**
     * @Route("/currencies/new", name="new_currency")
     * @IsGranted("ROLE_CURRENCY", message="Vous n'êtes pas autorisé à ajouter une monnaie")
     */
    public function new(Request $request): Response
    {
        return $this->edit('currencies/new.html.twig', $request);
    }

    /**
     * @Route("/currencies/{id}", name="details_currency")
     */
    public function details(Request $request, int $id): Response
    {
        return $this->edit('currencies/details.html.twig', $request, $id);
    }

    /**
     * @Route("/currencies/{id}/delete", name="delete_currency")
     */
    public function delete(Request $request, int $id): Response
    {
        return $this->crud($request, $id, Currency::class, [], '', []);
    }

    private function edit(string $template, Request $request, $id = null): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $entityRepository = $entityManager->getRepository(Currency::class);
        $currency = null;
        $violationList = [];
        $currency = \is_null($id) ? null : $entityRepository->find($id);

        if ($request->isMethod('POST')) {
            if ($request->isXmlHttpRequest()) {
                $payload = \json_decode(file_get_contents('php://input'), true);
                $code = $currency->getCode();
                $main = $payload['main'];
            } else {
                if (!isset($id)) {
                    if (!\is_null($currency)) {
                        throw new BadRequestException('The currency '.$currency->getCode().' already exists');
                    }
                    $currency = new Currency();
                }

                $code = $request->request->get('code');
                $main = $request->request->get('main');
            }

            $code = \strtoupper($code);

            if ($main === 'on') {
                $mainCurrency = $entityRepository->findOneBy(['main' => true]);

                if (!\is_null($mainCurrency)) {
                    $mainCurrency->setMain(false);
                }
            }

            if (!\in_array($code, Currencies::getCurrencyCodes())) {
                throw new NotFoundHttpException('The currency '.$code.' does not exist');
            }

            $currency->setMain($main === 'on' ? true : false);
            $currency->setCode($code);

            $violationList = $this->validator->validate($currency);
            $countViolations = \count($violationList);
            $this->get('session')->getFlashBag()->clear();

            if ($countViolations > 0) {
                $message = 'Le formulaire contient des erreurs';
                if ($request->isXMLHttpRequest()) {
                    return new ApiResponse($message, [], $violationList, 500);
                }
                $this->addFlash('danger', $message);
            } else {
                if (!isset($id)) {
                    $entityManager->persist($currency);
                }
                $entityManager->flush();
                if (isset($id)) {
                    $placeholder = 'modifiée';
                } else {
                    $placeholder = 'ajoutée';
                }
                if ($request->isXMLHttpRequest()) {
                    return new ApiResponse('', [], [], 200);
                }
                $this->addFlash('success', sprintf("Monnaie %s avec succès", $placeholder));
            }
        }

        return $this->render($template, [
            'title' => isset($id) ? 'Modifier monnaie' : 'Nouvelle monnaie',
            'breadcrumb' => [
                ['name' => 'Monnaies', 'path' => '/currencies'],
                ['name' => isset($id) ? $currency->getName() : 'Nouvelle monnaie']
            ],
            'latestCurrencies' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
            'currentC' => $currency,
            'violations' => $violationList
        ]);
    }
}
