<?php

namespace App\DataPersister;

use ApiPlatform\Core\DataPersister\ContextAwareDataPersisterInterface;
use App\Entity\Message;
use App\Tools\SmsUtils;

final class MessageDataPersister implements ContextAwareDataPersisterInterface
{
    private $decorated;
    private $smsUtils;

    public function __construct(ContextAwareDataPersisterInterface $decorated, SmsUtils $smsUtils)
    {
        $this->decorated = $decorated;
        $this->smsUtils = $smsUtils;
    }

    public function supports($data, array $context = []): bool
    {
        return $data instanceof Message;
    }

    public function persist($data, array $context = []): ?Message
    {
        $report = [];
        $clients = [];

        if (
            $data instanceof Message && (
                ($context['collection_operation_name'] ?? null) === 'post' ||
                ($context['collection_operation_name'] ?? null) === 'put' ||
                ($context['graphql_operation_name'] ?? null) === 'create'
            )
        ) {
            $clients = $data->getClients();
            $report = $this->sendMessage($data);
        }

        $previousClientCount = count($clients);
        $failedClients = [];

        foreach ($clients as $client) {
            $clientId = $client->getId();
            if (array_key_exists($clientId, $report) && $report[$clientId]['status'] === 0) {
                $failedClients[] = ['clientId' => $client->getId(), 'nomClient' => $client->getNomClient(), 'telephone' => $client->getTelephone()];
                $data->removeClient($client);
            }
        }

        $data->setFailedClients($failedClients);
        $clientCount = count($data->getClients());
        $data->setStatus(count($report) > 0 ? ($clientCount === $previousClientCount ? 2 : 1) : 0);
        $result = $this->decorated->persist($data, $context);

        return $result;
    }

    public function remove($data, array $context = [])
    {
        return $this->decorated->remove($data, $context);
    }

    private function sendMessage(Message $message)
    {
        $queue = [];
        $messageBody = $message->getBody();

        foreach ($message->getClients() as $client) {
            $queue[] = ['message' => $messageBody, 'clientId' => $client->getId(), 'nomClient' => $client->getNomClient(), 'telephone' => $client->getTelephone()];
        }

        return $this->smsUtils->sendMessage($queue);
    }
}
