<?php

namespace App\Controller;

use App\AbstractController;
use App\Entity\BankAccount;
use App\Entity\Currency;
use App\Http\ApiResponse;
use PhpOffice\PhpSpreadsheet\Calculation\DateTimeExcel\Current;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Intl\Currencies;
use Symfony\Component\Routing\Annotation\Route;

class BankAccountController extends AbstractController
{
    /**
     * @Route("/bankaccounts", name="bankaccounts_list")
     */
    public function list(): Response
    {
        $entityRepository = $this->getDoctrine()->getRepository(BankAccount::class);

        return $this->render('bankaccounts/list.html.twig', [
            'title' => 'Comptes bancaires',
            'breadcrumb' => [
                ['name' => 'Comptes bancaires']
            ],
            'bankaccounts' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
        ]);
    }

    /**
     * @Route("/bankaccounts/new", name="new_bankaccount")
     * @IsGranted("ROLE_FINANCE", message="Vous n'êtes pas autorisé à ajouter un compte bancaire")
     */
    public function new(Request $request): Response
    {
        return $this->edit('bankaccounts/new.html.twig', $request);
    }

    /**
     * @Route("/bankaccounts/{id}", name="details_bankaccount")
     */
    public function details(Request $request, int $id): Response
    {
        return $this->edit('bankaccounts/details.html.twig', $request, $id);
    }

    /**
     * @Route("/bankaccounts/{id}/delete", name="delete_bankaccount")
     */
    public function delete(Request $request, int $id): Response
    {
        return $this->crud($request, $id, BankAccount::class, [], '', []);
    }

    private function edit(string $template, Request $request, $id = null): Response
    {
        $entityManager = $this->getDoctrine()->getManager();
        $entityRepository = $entityManager->getRepository(BankAccount::class);
        $currencyRepository = $entityManager->getRepository(Currency::class);
        $bankaccount = null;
        $violationList = [];
        $bankaccount = \is_null($id) ? null : $entityRepository->find($id);
        $currencies = $currencyRepository->findAll();

        if ($request->isMethod('POST')) {
            if (!isset($id)) {
                $bankaccount = new BankAccount();
            }

            $number = $request->request->get('number');
            $title = $request->request->get('title');
            $bankName = $request->request->get('bankName');
            $currency = $currencyRepository->find(\intval($request->request->get('currency')));

            if (!\is_null($title)) {
                $title = \strtoupper($title);
                $bankaccount->setTitle($title);
            }

            $bankaccount->setNumber($number);
            $bankaccount->setBankName($bankName);
            $bankaccount->setCurrency($currency);

            $violationList = $this->validator->validate($bankaccount);
            $countViolations = \count($violationList);
            $this->get('session')->getFlashBag()->clear();

            if ($countViolations > 0) {
                $message = 'Le formulaire contient des erreurs';
                if ($request->isXMLHttpRequest()) {
                    return new ApiResponse($message, [], $violationList, 500);
                }
                $this->addFlash('danger', $message);
            } else {
                if (!isset($id)) {
                    $entityManager->persist($bankaccount);
                }
                $entityManager->flush();
                if (isset($id)) {
                    $placeholder = 'modifié';
                } else {
                    $placeholder = 'ajouté';
                }
                if ($request->isXMLHttpRequest()) {
                    return new ApiResponse('', [], [], 200);
                }
                $this->addFlash('success', sprintf("Compte bancaire %s avec succès", $placeholder));
            }
        }

        return $this->render($template, [
            'title' => isset($id) ? 'Modifier compte bancaire' : 'Nouveau compte bancaire',
            'breadcrumb' => [
                ['name' => 'Comptes bancaires', 'path' => '/bankaccounts'],
                ['name' => isset($id) ? $bankaccount->getNumber().' ('.$bankaccount->getBankName().')' : 'Nouveau compte bancaire']
            ],
            'latestBankAccounts' => $entityRepository->findBy([], ['createdAt' => 'DESC']),
            'currentB' => $bankaccount,
            'currencies' => $currencies,
            'violations' => $violationList
        ]);
    }
}
